# filepath: c:\Users\dpascoal\Documents\Projects\DHLAllinone\src\scripts\check-all-outlook-emails-for-tools.ps1
# Outlook Email Tool Scanner Script
# Author: Dario Pascoal
#
# Description: This PowerShell script scans a shared Outlook mailbox to find 
# emails related to specific tools (identified by Serial Numbers and Equipment 
# Numbers). It's designed for DHL's tool management workflow where emails 
# between DHL Service Tools and ASML Supplier Management need to be tracked 
# and associated with specific tool maintenance requests.
#
# Business Context: In the DHL tool management process, when tools are booked 
# for maintenance or inspection, there's often email correspondence with suppliers 
# like ASML. This script helps track that correspondence by:
# 1. Scanning emails in the shared DHL Service Tools mailbox
# 2. Looking for tool identifiers (Serial Numbers/Equipment Numbers) in email content
# 3. Filtering emails by date to only include relevant correspondence
# 4. Outputting structured data that can be used by the main application
#
# The script is particularly useful for:
# - Tracking supplier responses to tool maintenance requests
# - Maintaining audit trails of tool-related communications
# - Automatically associating emails with specific tools in the system
# - Providing data for reporting and compliance purposes
#
# Technical Implementation:
# - Uses Outlook COM Interop to access mailbox data
# - Applies date filtering to improve performance on large mailboxes
# - Implements retry logic for robust Outlook connectivity
# - Outputs results in JSON format for easy consumption by other applications
#
# Prerequisites:
# - Microsoft Outlook must be installed and configured
# - Access to the "DHL Service Tools" shared mailbox
# - Appropriate permissions to read mailbox contents
# - .NET Framework for COM Interop functionality
#
# Parameters:
# - ToolDataJson: JSON string containing tool information with Serial Numbers and Equipment Numbers
# - OutputJsonPath: Path where the results JSON file will be saved
# - AppLogFilePath: Optional path for application logging
#
# Usage Example:
# .\check-all-outlook-emails-for-tools.ps1 -ToolDataJson $toolData -OutputJsonPath "C:\temp\results.json"

param(
    [Parameter(Mandatory=$true)][string]$ToolDataJson,
    [Parameter(Mandatory=$false)][string]$OutputJsonPath = (Join-Path $env:TEMP "email_results.json"), # Default to temp directory
    [Parameter(Mandatory=$false)][string]$AppLogFilePath # Optional: For consistency if needed later
)

# =============================================================================
# SECTION 1: CONFIGURATION AND CONSTANTS
# =============================================================================
# This section defines the core configuration parameters that control how the
# script operates. These values are specific to DHL's email infrastructure
# and the ASML supplier relationship.

# --- Configuration ---
$SharedMailboxName = "DHL Service Tools"                                    # The name of the shared mailbox to scan
$TargetEmailAddress = "asml_tc-ts-suppliermanagement@asml.com"             # ASML supplier email (lowercase for comparison)
$TargetDisplayName = "ASML_TC-TS Supplier Management"                       # ASML supplier display name
$MaxRetryAttempts = 3                                                       # Number of retry attempts for Outlook operations
$SearchLookbackDays = 90                                                    # Default lookback period if no tool dates available
# --- End Configuration ---

# =============================================================================
# SECTION 2: LOGGING AND OUTPUT HELPER FUNCTIONS
# =============================================================================
# These functions provide consistent logging and output formatting throughout
# the script. They handle both console output (for real-time monitoring) and
# file logging (for audit trails and debugging).

# --- Helper Functions ---
function Write-DebugLog {
    param([string]$Message)
    # Create timestamp for log entries - helps with debugging and audit trails
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logEntry = "$timestamp - DEBUG [ALL-EMAILS]: $Message"
    Write-Host $logEntry # Output for Electron process capture

    # Also write to application log file if path was provided
    # This provides persistent logging for troubleshooting issues
    if (![string]::IsNullOrEmpty($AppLogFilePath)) {
        try {
            Add-Content -Path $AppLogFilePath -Value "[$timestamp] [DEBUG] [ALL-OUTLOOK-EMAILS] $Message" -Encoding UTF8 -ErrorAction SilentlyContinue
        } catch {
            # Silently ignore log file write errors for this script
            # We don't want logging issues to break the main functionality
        }
    }
}

function Write-ScriptResult {
    param(
        [Parameter(Mandatory=$true)][bool]$Success,
        [Parameter(Mandatory=$false)][string]$Message
    )
    # Output results in a standardized format that the parent Electron process can parse
    # This structured output allows the main application to determine script success/failure
    Write-Host "SCRIPT_RESULT_START"
    Write-Host "Success=$Success"
    if ($Message) {
        Write-Host "Message=$Message"
    }
    Write-Host "SCRIPT_RESULT_END"
}

# =============================================================================
# SECTION 3: MAIN SCRIPT INITIALIZATION AND VARIABLE SETUP
# =============================================================================
# This section initializes the main variables used throughout the script and
# sets up the data structures needed for processing emails and storing results.

# --- Main Script Logic ---
$Outlook = $null                                                            # Will hold the Outlook COM object
$Namespace = $null                                                          # Will hold the MAPI namespace object
$SharedInbox = $null                                                        # Will hold the shared mailbox Inbox folder
$SharedSentItems = $null                                                    # Will hold the shared mailbox Sent Items folder
$allRelevantEmails = New-Object System.Collections.Generic.List[System.Object] # Store raw email matches before filtering
$finalResults = @{}                                                         # Hashtable to store final results grouped by Tool ID (SN/EQ)

try {
    Write-DebugLog "--- Starting Bulk Email Check ---"
    Write-DebugLog "Output JSON Path: $OutputJsonPath"

    # =============================================================================
    # SECTION 4: TOOL DATA PARSING AND LOOKUP TABLE CREATION
    # =============================================================================
    # This section parses the input JSON containing tool information and creates
    # efficient lookup structures. It also determines the earliest date for
    # filtering emails to improve performance.

    # 1. Parse Input Tool Data
    Write-DebugLog "Parsing ToolDataJson..."
    $tools = $null
    try {
        # Convert the JSON string to PowerShell objects for easier processing
        $tools = $ToolDataJson | ConvertFrom-Json -ErrorAction Stop
        if ($null -eq $tools -or $tools.Count -eq 0) {
            throw "Input ToolDataJson was empty or invalid."
        }
        
        # Create lookup tables for faster checking during email processing
        # Using hashtables provides O(1) lookup performance instead of O(n) array searching
        $toolLookup = @{}
        $earliestBookedDate = [DateTime]::MaxValue
        
        # Process each tool and create lookup entries
        foreach ($tool in $tools) {
            $bookedDate = $null
            
            # Parse the tool's booking/change date if it exists and is in the expected format (YYYYMMDD)
            if (![string]::IsNullOrWhiteSpace($tool.ChangedDate) -and $tool.ChangedDate -match '^\d{8}$') {
                try {
                    $bookedDate = [DateTime]::ParseExact($tool.ChangedDate, 'yyyyMMdd', $null)
                    # Keep track of the earliest date for Outlook filtering optimization
                    if ($bookedDate -lt $earliestBookedDate) {
                        $earliestBookedDate = $bookedDate
                    }
                } catch {
                    Write-DebugLog "Warning: Could not parse ChangedDate '$($tool.ChangedDate)' for tool SN:$($tool.SerialNumber)/EQ:$($tool.EquipmentNumber). Skipping date filter for this tool."
                    $bookedDate = $null # Ensure it's null if parse fails
                }
            } else {
                 Write-DebugLog "Warning: Invalid or missing ChangedDate '$($tool.ChangedDate)' for tool SN:$($tool.SerialNumber)/EQ:$($tool.EquipmentNumber). Skipping date filter for this tool."
                 $bookedDate = $null
            }

            # Create a tool entry with all relevant information
            $toolEntry = @{ 
                BookedDate = $bookedDate
                SerialNumber = $tool.SerialNumber
                EquipmentNumber = $tool.EquipmentNumber 
            }

            # Add entries to lookup table for both Serial Number and Equipment Number
            # We use lowercase keys for case-insensitive matching in emails
            if (![string]::IsNullOrWhiteSpace($tool.SerialNumber)) {
                $toolLookup[$tool.SerialNumber.ToLower()] = $toolEntry # Use lowercase keys
            }
            if (![string]::IsNullOrWhiteSpace($tool.EquipmentNumber)) {
                $toolLookup[$tool.EquipmentNumber.ToLower()] = $toolEntry # Use lowercase keys
            }
        }
        Write-DebugLog "Parsed $($tools.Count) tools. Found $($toolLookup.Count) unique SN/EQ identifiers."
        Write-DebugLog "Earliest BookedDate found: $($earliestBookedDate.ToString('yyyy-MM-dd'))"

    } catch {
        Write-DebugLog "FATAL: Failed to parse ToolDataJson: $($_.Exception.Message)"
        Write-ScriptResult -Success $false -Message "Invalid input tool data."
        exit 1 # Use non-zero exit code for fatal input error
    }

    # =============================================================================
    # SECTION 5: OUTLOOK CONNECTION AND COM INTEROP SETUP
    # =============================================================================
    # This section establishes a connection to Microsoft Outlook using COM Interop.
    # It includes retry logic to handle common Outlook connectivity issues and
    # attempts to connect to an existing instance before creating a new one.

    # 2. Connect to Outlook
    Write-DebugLog "Connecting to Outlook..."
    try {
        # Load the Outlook Interop Assembly for COM communication
        # This assembly provides the .NET interface to Outlook's COM objects
        Add-Type -assembly "Microsoft.Office.Interop.Outlook" -ErrorAction Stop
    } catch {
        Write-DebugLog "FATAL: Failed to load Outlook Interop Assembly: $($_.Exception.Message)"
        Write-ScriptResult -Success $false -Message "Outlook Interop Assembly not found."
        exit 1
    }

    # Implement retry logic for Outlook connection
    # Outlook connections can be temperamental, especially in automated environments
    $retryCount = 0
    $outlookConnected = $false
    while (-not $outlookConnected -and $retryCount -lt $MaxRetryAttempts) {
        try {
            # Try to get an existing Outlook instance first (preferred approach)
            # This avoids starting multiple Outlook processes and is faster
            $Outlook = [System.Runtime.InteropServices.Marshal]::GetActiveObject('Outlook.Application')
            $outlookConnected = $true
            Write-DebugLog "Connected to running Outlook instance."
        } catch {
            $retryCount++
            Write-DebugLog "Failed to get active Outlook (Attempt $retryCount/$MaxRetryAttempts): $($_.Exception.Message)"
            if ($retryCount -ge $MaxRetryAttempts) {
                try {
                    # If no existing instance, try to start a new Outlook instance
                    Write-DebugLog "Trying to start new Outlook instance..."
                    $Outlook = New-Object -ComObject Outlook.Application
                    Start-Sleep -Seconds 5  # Give Outlook time to fully initialize
                    $outlookConnected = $true
                    Write-DebugLog "New Outlook instance started."
                } catch {
                     Write-DebugLog "FATAL: Failed to start or connect to Outlook: $($_.Exception.Message)"
                     Write-ScriptResult -Success $false -Message "Could not connect to Outlook."
                     exit 1
                }
            } else {
                Start-Sleep -Seconds 2  # Wait before retrying
            }
        }
    }

    # Get the MAPI namespace - this is the entry point for accessing Outlook data
    # MAPI (Messaging Application Programming Interface) provides access to mailboxes, folders, and items
    $Namespace = $Outlook.GetNamespace("MAPI")

    # =============================================================================
    # SECTION 6: SHARED MAILBOX ACCESS AND FOLDER NAVIGATION
    # =============================================================================
    # This section accesses the shared DHL Service Tools mailbox and navigates
    # to the required folders (Inbox and Sent Items). It includes multiple 
    # fallback strategies for different Outlook configurations and permissions.

    # 3. Access Shared Mailbox Folders
    Write-DebugLog "Accessing shared mailbox '$SharedMailboxName'..."
    $mailboxAccessSuccess = $false
    $retryCount = 0
    
    # Implement retry logic for mailbox access as this can fail due to various reasons:
    # - Network connectivity issues
    # - Mailbox permissions
    # - Outlook synchronization delays
    while (-not $mailboxAccessSuccess -and $retryCount -lt $MaxRetryAttempts) {
         try {
            $currentMailboxFolder = $null
            
            # Method 1: Try to find the mailbox by name in the folder collection
            # This works when the shared mailbox is already loaded in the Outlook profile
            $Namespace.Folders | ForEach-Object {
                if ($_.Name -eq $SharedMailboxName) {
                    $currentMailboxFolder = $_
                    Write-DebugLog "Found shared mailbox by name: $SharedMailboxName"
                }
            }

            # Method 2: If direct access failed, try recipient resolution approach
            # This works for shared mailboxes that aren't directly visible in the folder list
            if ($null -eq $currentMailboxFolder) {
                 $availableMailboxes = $Namespace.Folders | ForEach-Object { $_.Name }
                 Write-DebugLog "Shared mailbox '$SharedMailboxName' not found directly. Available: $($availableMailboxes -join ', ')"
                 
                 # Create a recipient object and resolve it to get mailbox access
                 # This is often necessary for shared mailboxes that require explicit access
                 $recipient = $Namespace.CreateRecipient($SharedMailboxName)
                 $recipient.Resolve()
                 if ($recipient.Resolved) {
                     Write-DebugLog "Recipient '$SharedMailboxName' resolved. Attempting GetSharedDefaultFolder."
                     
                     # Try getting Inbox first (most common scenario)
                     try {
                         $SharedInbox = $Namespace.GetSharedDefaultFolder($recipient, [Microsoft.Office.Interop.Outlook.OlDefaultFolders]::olFolderInbox)
                         $currentMailboxFolder = $SharedInbox.Parent # Get the root folder of the shared mailbox
                         Write-DebugLog "Accessed shared mailbox '$($currentMailboxFolder.Name)' via resolved recipient (Inbox)."
                     } catch {
                         Write-DebugLog "Could not get shared Inbox via recipient. Trying Sent Items..."
                         # Try getting Sent Items if Inbox failed (alternative access method)
                         try {
                            $SharedSentItems = $Namespace.GetSharedDefaultFolder($recipient, [Microsoft.Office.Interop.Outlook.OlDefaultFolders]::olFolderSentMail)
                            $currentMailboxFolder = $SharedSentItems.Parent
                            Write-DebugLog "Accessed shared mailbox '$($currentMailboxFolder.Name)' via resolved recipient (Sent Items)."
                         } catch {
                             Write-DebugLog "Could not get shared Sent Items via recipient either."
                             throw "Could not access shared mailbox '$SharedMailboxName' via recipient resolution."
                         }
                     }
                 } else {
                     throw "Could not resolve recipient '$SharedMailboxName'."
                 }
            }

            # Navigate to required folders within the mailbox
            # We need both Inbox (for received emails) and Sent Items (for outgoing emails)
            if ($null -eq $SharedInbox) { # If not already set via GetSharedDefaultFolder
                # Look for Inbox folder with language-aware name matching
                $SharedInbox = $currentMailboxFolder.Folders | Where-Object { $_.Name -match 'Inbox|Posteingang' } | Select-Object -First 1
                if ($null -eq $SharedInbox) { $SharedInbox = $currentMailboxFolder.Folders.Item("Inbox") } # Fallback to exact name
            }
            if ($null -eq $SharedSentItems) { # If not already set via GetSharedDefaultFolder
                # Look for Sent Items folder with language-aware name matching
                $SharedSentItems = $currentMailboxFolder.Folders | Where-Object { $_.Name -match 'Sent Items|Gesendete Elemente' } | Select-Object -First 1
                if ($null -eq $SharedSentItems) { $SharedSentItems = $currentMailboxFolder.Folders.Item("Sent Items") } # Fallback to exact name
            }

            # Verify we successfully found the required folders
            if ($null -eq $SharedInbox) { throw "Could not find Inbox folder in mailbox '$($currentMailboxFolder.Name)'." }
            if ($null -eq $SharedSentItems) { throw "Could not find Sent Items folder in mailbox '$($currentMailboxFolder.Name)'." }

            Write-DebugLog "Using mailbox: $($currentMailboxFolder.Name)"
            Write-DebugLog "Using Inbox folder: $($SharedInbox.Name)"
            Write-DebugLog "Using Sent Items folder: $($SharedSentItems.Name)"
            $mailboxAccessSuccess = $true

         } catch {
            $retryCount++
            Write-DebugLog "Error accessing mailbox (Attempt $retryCount/$MaxRetryAttempts): $($_.Exception.Message)"
            if ($retryCount -ge $MaxRetryAttempts) {
                Write-DebugLog "FATAL: Error accessing mailbox '$SharedMailboxName': $($_.Exception.Message)"
                Write-ScriptResult -Success $false -Message "Error accessing mailbox '$SharedMailboxName'."
                exit 1
            }
            Start-Sleep -Seconds 3  # Wait before retrying
         }
    }

    # =============================================================================
    # SECTION 7: EMAIL SCANNING SETUP AND DATE FILTERING
    # =============================================================================
    # This section prepares for the email scanning process by setting up date
    # filters and initializing counters. The date filtering is crucial for
    # performance when dealing with large mailboxes.

    # 4. Scan Emails and Collect Raw Matches
    Write-DebugLog "Scanning emails (lookback approx $($SearchLookbackDays) days)..."
    $processedCount = 0      # Counter for total emails processed
    $rawMatchCount = 0       # Counter for emails that match our criteria before date filtering
    $scanStartTime = Get-Date # Track scanning performance

    # Determine the optimal date filter for Outlook queries
    # This significantly improves performance by limiting the initial email set
    $filterDateForOutlook = $null
    if ($earliestBookedDate -ne [DateTime]::MaxValue) {
        # If we have valid tool booking dates, use the earliest one as our filter
        # This ensures we don't miss any relevant emails while optimizing performance
        $filterDateForOutlook = $earliestBookedDate
        Write-DebugLog "Using earliest BookedDate from tools for initial Outlook filter: $($filterDateForOutlook.ToString('yyyy-MM-dd'))"
    } else {
        # Fallback to a general lookback period if no tool dates were found
        # This prevents scanning the entire mailbox history unnecessarily
        $filterDateForOutlook = (Get-Date).AddDays(-$SearchLookbackDays)
        Write-DebugLog "No valid BookedDates found in tools. Using general lookback of $SearchLookbackDays days for initial Outlook filter: $($filterDateForOutlook.ToString('yyyy-MM-dd'))"
    }
    
    # Format the date for Outlook's restriction filter
    # Outlook requires MM/dd/yyyy format with en-US culture for reliable parsing
    $restrictionDateString = $filterDateForOutlook.ToString("MM/dd/yyyy hh:mm tt", [System.Globalization.CultureInfo]::GetCultureInfo('en-US'))
    $filter = "([ReceivedTime] >= '$restrictionDateString' OR [SentOn] >= '$restrictionDateString')"
    Write-DebugLog "Applying Outlook filter: $filter"

    # =============================================================================
    # SECTION 8: INBOX EMAIL PROCESSING AND PATTERN MATCHING
    # =============================================================================
    # This section processes emails in the Inbox folder, applying complex business
    # logic to determine which emails are relevant to our tool management process.
    # It includes sophisticated pattern matching and relationship detection.

    # Process Inbox
    $inboxItems = $null
    try {
        Write-DebugLog "Processing Inbox '$($SharedInbox.Name)'..."
        
        # Apply the date filter to get a manageable set of emails
        # This uses Outlook's native filtering which is much faster than processing all emails
        $inboxItems = $SharedInbox.Items.Restrict($filter)
        $inboxItems.Sort("[ReceivedTime]", $true) # Sort descending (newest first) for better user experience
        Write-DebugLog "Found approx $($inboxItems.Count) items in Inbox matching date filter."

        # Process each email item in the filtered set
        foreach ($item in $inboxItems) {
            $processedCount++
            $itemSubject = "N/A"  # Default value in case subject access fails
            try {
                # Extract key email properties for analysis
                # These properties help us determine email relevance and relationships
                $itemSubject = $item.Subject
                $itemBody = $item.Body
                $itemSenderAddress = $item.SenderEmailAddress.ToLower() # Normalize for comparison
                $itemSenderName = $item.SenderName
                
                # Build recipient list for comprehensive relationship analysis
                # We need to check both TO and CC recipients to understand email context
                $itemRecipients = ""
                foreach ($recipient in $item.Recipients) {
                    try {
                        $itemRecipients += $recipient.Address.ToLower() + ";" # Normalize and concatenate
                    } catch {
                         # Ignore errors getting individual recipient addresses
                         # Some recipient objects may be malformed or inaccessible
                    }
                }
                
                # Extract timing information for date-based filtering
                $itemReceivedTime = $item.ReceivedTime
                $itemSentOn = $item.SentOn # Keep SentOn in case ReceivedTime is null
                $itemEntryID = $item.EntryID           # Unique identifier for this email
                $itemConversationID = $item.ConversationID # Groups related emails together

                # ==========================================================================
                # BUSINESS LOGIC: EMAIL RELEVANCE DETERMINATION
                # ==========================================================================
                # This section implements the core business logic for determining which
                # emails are relevant to our tool management process. It checks multiple
                # criteria to identify emails that involve the ASML supplier relationship.

                # Check 1: Does this email involve the target ASML email/display name?
                # We need to identify emails that are part of the DHL-ASML communication chain
                $involvesTarget = $false
                $reasonIncluded = ""
                
                # Primary check: Direct communication with ASML target
                # These are the most obvious cases of relevant emails
                if ($itemSenderAddress -like "*$TargetEmailAddress*") {
                    $involvesTarget = $true
                    $reasonIncluded = "Sender address matches ASML target"
                }
                elseif ($itemRecipients -like "*$TargetEmailAddress*") {
                    $involvesTarget = $true
                    $reasonIncluded = "Recipients include ASML target"
                }
                elseif ($itemSenderName -ilike "*$TargetDisplayName*") {
                    $involvesTarget = $true
                    $reasonIncluded = "Sender name matches ASML target"
                }
                
                # Secondary check: Emails involving our shared mailbox
                # This is CRITICAL for catching outgoing emails and internal communications
                # that might not directly involve the ASML target but are still relevant
                if (-not $involvesTarget) {
                    # Check for various forms of our shared mailbox identification
                    # Email systems use different addressing formats depending on configuration
                    if ($itemRecipients -like "*dhl.service.tools@asml.com*") {
                        $involvesTarget = $true
                        $reasonIncluded = "Shared mailbox email found in recipients"
                    }
                    elseif ($itemRecipients -like "*$SharedMailboxName*") {
                        $involvesTarget = $true
                        $reasonIncluded = "Shared mailbox display name found in recipients"
                    }
                    # Check for internal Exchange addressing patterns
                    # Exchange often uses internal addressing that doesn't match display names
                    elseif ($itemRecipients -like "*dhl service*") {
                        $involvesTarget = $true
                        $reasonIncluded = "Shared mailbox name pattern found in recipients"
                    }
                    elseif ($itemRecipients -like "*75fa2d0a9451469b8012d203ba33c5a3-dhl service*") {
                        $involvesTarget = $true
                        $reasonIncluded = "Shared mailbox Exchange ID found in recipients"
                    }
                    # Check for emails sent FROM this mailbox (appears in Inbox due to rules/forwarding)
                    elseif ($itemSenderAddress -like "*dhl.service.tools@asml.com*" -or $itemSenderName -like "*DHL Service Tools*") {
                        $involvesTarget = $true
                        $reasonIncluded = "Sent from shared mailbox"
                    }
                    # Fallback: Any email containing tool identifiers that's DHL-related
                    elseif ($containsIdentifiers) {
                        $involvesTarget = $true
                        $reasonIncluded = "Contains tool identifiers and related to DHL service"
                    }
                }

                # ==========================================================================
                # TOOL IDENTIFIER DETECTION AND DEBUGGING
                # ==========================================================================
                # This section searches email content for tool identifiers (Serial Numbers
                # and Equipment Numbers) and provides comprehensive debugging information
                # to help troubleshoot why emails might not be included in results.

                # Scan email content for any tool identifiers from our lookup table
                # This is done before the target involvement check for comprehensive debugging
                $containsIdentifiers = $false
                $foundIdentifiersForDebug = New-Object System.Collections.Generic.List[string]
                
                # Search through all tool identifiers to see if any appear in this email
                foreach ($key in $toolLookup.Keys) {
                    # Use regex escaping to handle special characters in tool identifiers
                    # This ensures that identifiers with special regex characters are matched literally
                    $pattern = [regex]::Escape($key)
                    if (($itemSubject -match $pattern) -or ($itemBody -match $pattern)) {
                        $containsIdentifiers = $true
                        $foundIdentifiersForDebug.Add($key)
                    }
                }
                
                # CRITICAL DEBUGGING: Log emails with tool identifiers for troubleshooting
                # This helps identify emails that contain relevant tools but aren't being processed
                if ($containsIdentifiers) {
                    # Only debug emails that contain tool identifiers to minimize log noise
                    if ($involvesTarget) {
                        Write-DebugLog "MATCH: Email subject '$itemSubject' included: $reasonIncluded"
                    } else {
                        # This is critical debugging for emails that contain tools but aren't included
                        # These logs help identify configuration issues or missing business logic
                        Write-DebugLog "NOT MATCHED: Email subject '$itemSubject' contains tools ${foundIdentifiersForDebug} but was NOT included"
                        Write-DebugLog "   Sender: '$itemSenderName' <$itemSenderAddress>"
                        Write-DebugLog "   Recipients: $itemRecipients"
                    }
                }

                if ($involvesTarget) {
                    # Check 1: Does it contain any SN/EQ from our list?
                    $foundIdentifiers = New-Object System.Collections.Generic.List[string]
                    $foundIdentifiersForDebug = New-Object System.Collections.Generic.List[string]
                    $containsIdentifiers = $false
                    
                    foreach ($key in $toolLookup.Keys) {
                        # Use regex matching for robustness (escape key)
                        $pattern = [regex]::Escape($key)
                        if (($itemSubject -match $pattern) -or ($itemBody -match $pattern)) {
                            $foundIdentifiers.Add($key) # Add the matched key (lowercase SN/EQ)
                            $foundIdentifiersForDebug.Add($key)
                            $containsIdentifiers = $true
                        }
                    }
                    
                    # STRICT EMAIL MATCHING LOGIC PER CLIENT REQUIREMENTS:
                    $emailMatches = $false
                    $reasonIncluded = ""
                    
                    # Get raw To and CC fields as they are most reliable
                    $rawTo = ""
                    $rawCC = ""
                    try { $rawTo = $item.To } catch {}
                    try { $rawCC = $item.CC } catch {}
                    
                    # Convert everything to lowercase for case-insensitive comparison
                    $toLower = $rawTo.ToLower()
                    $ccLower = $rawCC.ToLower() 
                    $senderAddressLower = $itemSenderAddress.ToLower()
                    $targetEmailLower = "asml_tc-ts-suppliermanagement@asml.com"
                    $sharedMailboxLower = "dhl.service.tools@asml.com"
                    
                    # CASE 1: INCOMING EMAIL (A REPLY) - FROM THE TARGET EMAIL AND SHARED MAILBOX IN TO OR CC
                    if ($senderAddressLower -like "*$targetEmailLower*" -and 
                        ($toLower -like "*$sharedMailboxLower*" -or $ccLower -like "*$sharedMailboxLower*")) {
                        $emailMatches = $true
                        $reasonIncluded = "Received from ASML SupplierManagement with our mailbox in TO/CC"
                    }
                    
                    # CASE 2: OUTGOING EMAIL - TO TARGET EMAIL AND CC'D TO SHARED MAILBOX
                    elseif (($toLower -like "*$targetEmailLower*") -and ($ccLower -like "*$sharedMailboxLower*")) {
                        $emailMatches = $true
                        $reasonIncluded = "Sent to ASML SupplierManagement with shared mailbox CC'd"
                    }
                    
                    # Debug logging for emails that contain tool identifiers
                    if ($containsIdentifiers) {
                        if ($emailMatches) {
                            Write-DebugLog "MATCH: Email subject '$itemSubject' included: $reasonIncluded"
                        } else {
                            Write-DebugLog "NOT MATCHED: Email subject '$itemSubject' contains tools $($foundIdentifiersForDebug -join ' ') but was NOT included"
                            Write-DebugLog "   Sender: '$itemSenderName' <$senderAddressLower>"
                            Write-DebugLog "   To: $toLower"
                            Write-DebugLog "   CC: $ccLower"
                        }
                    }

                    # Add if it matches identifiers (NO check for IsReply needed here)
                    if ($foundIdentifiers.Count -gt 0) {
                        $rawMatchCount++
                        # Store relevant data
                        $recipientNames = ($item.Recipients | ForEach-Object { $_.Name }) -join '; '
                        $emailData = [PSCustomObject]@{
                            Subject = $itemSubject
                            Body = $itemBody
                            Sender = $itemSenderName
                            Recipients = $recipientNames
                            Date = if ($itemReceivedTime) { $itemReceivedTime.ToString("yyyy-MM-dd HH:mm") } elseif ($itemSentOn) { $itemSentOn.ToString("yyyy-MM-dd HH:mm") } else { "Unknown Date" }
                            DateTimeObj = if ($itemReceivedTime) { $itemReceivedTime } elseif ($itemSentOn) { $itemSentOn } else { $null }
                            IsReply = ($itemSubject -match '^(RE|AW|WG|FW|VS|VB):') # Still determine if it IS a reply
                            IsSentItem = $false # Always false now
                            FolderName = $SharedInbox.Name
                            EntryID = $itemEntryID
                            ConversationID = $itemConversationID
                            MatchedIDs = $foundIdentifiers
                        }
                        $allRelevantEmails.Add($emailData)
                    }
                }

            } catch {
                Write-DebugLog "Error processing item #$processedCount (Subject: '$itemSubject'): $($_.Exception.Message)"
            } finally {
                if ($null -ne $item) { try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($item) | Out-Null } catch {} }
            }
        }
        Write-DebugLog "Finished Inbox. Processed $processedCount items total. Found $rawMatchCount raw matches (from Inbox only)." # Updated log message

    } catch {
         Write-DebugLog "ERROR processing Inbox items: $($_.Exception.Message)"
    } finally {
         if ($null -ne $inboxItems) { try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($inboxItems) | Out-Null } catch {} }
    }

    # --- SENT ITEMS PROCESSING BLOCK REMAINS REMOVED ---

    $scanEndTime = Get-Date
    Write-DebugLog "Email scanning finished in $(($scanEndTime - $scanStartTime).TotalSeconds) seconds."

    # 5. Filter Raw Matches by BookedDate and Group by Tool ID
    Write-DebugLog "Filtering $rawMatchCount raw matches against tool BookedDates..."
    $finalMatchCount = 0

    # Initialize finalResults with all tool keys from lookup
    foreach ($key in $toolLookup.Keys) {
        $finalResults[$key] = New-Object System.Collections.Generic.List[System.Object]
    }

    foreach ($email in $allRelevantEmails) {
        if ($null -eq $email.DateTimeObj) {
            Write-DebugLog "Skipping email (Subject: '$($email.Subject)') due to missing DateTimeObj."
            continue # Skip if date couldn't be parsed/stored
        }

        foreach ($matchedId in $email.MatchedIDs) { # Iterate through SN/EQ keys found in this email
            if ($toolLookup.ContainsKey($matchedId)) {
                $toolInfo = $toolLookup[$matchedId]
                # Check if email date is >= tool's booked date (if booked date is valid)
                if ($null -eq $toolInfo.BookedDate -or $email.DateTimeObj -ge $toolInfo.BookedDate) {
                    # Add a *copy* of the email data, removing MatchedIDs and DateTimeObj
                    $outputEmailData = $email | Select-Object * -ExcludeProperty MatchedIDs, DateTimeObj
                    $finalResults[$matchedId].Add($outputEmailData)
                    $finalMatchCount++
                    # Write-DebugLog "Final Match: Email '$($email.Subject)' added to tool '$matchedId' (Date $($email.Date) >= BookedDate $($toolInfo.BookedDate))" # Too verbose
                } else {
                    # Write-DebugLog "Filtered out: Email '$($email.Subject)' for tool '$matchedId' (Date $($email.Date) < BookedDate $($toolInfo.BookedDate))" # Too verbose
                }
            } else {
                 Write-DebugLog "Warning: Matched ID '$matchedId' from email not found in tool lookup (should not happen)."
            }
        }
    }
    Write-DebugLog "Filtering complete. Found $finalMatchCount relevant email instances across all tools."

    # 6. Convert Final Results to JSON and Save
    Write-DebugLog "Converting final results to JSON..."
    $jsonOutput = $null
    try {
        # Convert the hashtable $finalResults to JSON
        # Need to handle the List[Object] inside the hashtable correctly
        $jsonOutput = $finalResults | ConvertTo-Json -Depth 5 -Compress -ErrorAction Stop
        Write-DebugLog "JSON conversion successful."
    } catch {
        Write-DebugLog "FATAL: Failed to convert results to JSON: $($_.Exception.Message)"
        Write-ScriptResult -Success $false -Message "Failed to generate JSON output."
        exit 1
    }

    Write-DebugLog "Saving JSON to '$OutputJsonPath'..."
    try {
        # Ensure directory exists
        $outputDir = Split-Path -Path $OutputJsonPath -Parent
        if (-not (Test-Path -Path $outputDir)) {
            New-Item -ItemType Directory -Path $outputDir -Force | Out-Null
            Write-DebugLog "Created output directory: $outputDir"
        }
        # Write JSON content WITHOUT BOM
        # Use [System.IO.File]::WriteAllText for better control over encoding/BOM
        $utf8EncodingWithoutBom = New-Object System.Text.UTF8Encoding($false) # $false specifies no BOM
        [System.IO.File]::WriteAllText($OutputJsonPath, $jsonOutput, $utf8EncodingWithoutBom)
        # Set-Content -Path $OutputJsonPath -Value $jsonOutput -Encoding UTF8 -ErrorAction Stop # <-- Replaced this line
        Write-DebugLog "Successfully saved JSON output."
        Write-ScriptResult -Success $true -Message "Email check complete. Results saved to $OutputJsonPath"

    } catch {
        Write-DebugLog "FATAL: Failed to write JSON to '$OutputJsonPath': $($_.Exception.Message)"
        Write-ScriptResult -Success $false -Message "Failed to save JSON output file."
        exit 1
    }

} catch {
    # Catch any unexpected top-level errors
    Write-DebugLog "FATAL SCRIPT ERROR: $($_.Exception.ToString())"
    Write-ScriptResult -Success $false -Message "Bulk email check encountered a fatal script error."
    exit 1 # Use non-zero exit code for fatal script errors
} finally {
    # --- COM Object Cleanup ---
    Write-DebugLog "Cleaning up COM objects..."
    if ($null -ne $SharedSentItems) { try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($SharedSentItems) | Out-Null } catch {} }
    if ($null -ne $SharedInbox) { try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($SharedInbox) | Out-Null } catch {} }
    if ($null -ne $currentMailboxFolder) { try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($currentMailboxFolder) | Out-Null } catch {} }
    if ($null -ne $Namespace) { try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($Namespace) | Out-Null } catch {} }
    if ($null -ne $Outlook) { try { [System.Runtime.InteropServices.Marshal]::ReleaseComObject($Outlook) | Out-Null } catch {} }
    [System.GC]::Collect()
    [System.GC]::WaitForPendingFinalizers()
    Write-DebugLog "--- Bulk Email Check Script Finished ---"
}
